# Import packages
import mph
import numpy as np
import pandas as pd
import glob

# Connect to server and load model file
base_dir = "C:/Users/Mark/OneDrive - AltaSim Technologies/Documents/Tips and Tricks/Jan2026 Python mph module/"
file = "busbar.mph"

# start client - primary method
client1 = mph.start()

# start client - back up method
#server1 = mph.Server()                      # Start server
#client1 = mph.Client(port=server1.port)     # Connect to server

# Load model
model = client1.load(base_dir+file)         # Load model

# Model Tree
mph.tree(model)

# Parameters
model.parameters()
model.parameter('Vtot')
model.parameter('Vtot','40[mV]')
model.parameters()

# Variables
model.java.component("comp1").cpl().create("intop1", "Integration") # Define Integration operator
model.java.component("comp1").cpl("intop1").selection().all()
model.java.component("comp1").cpl().create("maxop1", "Maximum")     # Define maximum operator
model.java.component("comp1").cpl("maxop1").selection().all()

model.java.component("comp1").variable().create("var1")             # Create variables node
model.java.component("comp1").variable("var1").set("Etot", "mat1.def.rho*mat1.def.Cp*intop1(T-293.15)") # Variable for total thermal energy
model.java.component("comp1").variable("var1").set("Tmax", "maxop1(T)");    # Variable for maximum temperature

# Sweep over voltages and compile results in pandas dataframe
Vtot = [10,15,20,25,30,35,40]                           # List of voltages to sweep over
df = pd.DataFrame(columns=['Vtot', 'Tmax', 'Etot'])     # Initialize dataframe
for V in Vtot:
    model.parameter('Vtot',str(V)+'[mV]')               # Update Vtot parameter in model
    model.solve()                                       # Solve model
    Tmax = model.evaluate('Tmax', 'degC')               # Evaluate variables for max temperature and 
    Etot = model.evaluate('Etot', 'kJ')                 # total energy
    row_data = [V,Tmax,Etot]                            # data for new row in dataframe
    df.loc[len(df)] = row_data                          # Add row to dataframe
    filename = 'busbar_'+str(V)+'mV.mph'                # Save separate mph file for current parameter case
    model.save(base_dir+filename)
client1.remove(model)
overall_maxT = df['Tmax'].max()
overall_maxT = overall_maxT.item()

# Loop over COMSOL files and export plot from each
for fname in glob.glob('*.mph'):
    model = client1.load(base_dir+fname)                        # Load model
    model.java.result("pg3").feature("vol1").set("unit", "\u00b0C")  # Set units to deg C in T plot
    model.java.result("pg3").feature("vol1").set("rangecolormax", overall_maxT)   # Set maximum on color scale
    model.java.result().export().create("img1", "Image")        # Create image export node
    model.java.result().export("img1").set("sourceobject", "pg3");
    imgfile = 'TempVolPlot_'+fname+'.png'
    model.java.result().export("img1").set("pngfilename", imgfile);
    model.java.result().export("img1").run();
    
    
